/**
* \file: app_iface_implv2.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: automounter
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2010, 2011 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#include <string.h>

#include "app_iface/app_iface_extern.h"

#include "ipc/info_messages.h"

#include "app_iface/filedb_writer.h"
#include "app_iface/message_dispatcher.h"
#include "app_iface/message_sender.h"
#include "model/device.h"
#include "model/partition.h"
#include "utils/logger.h"

//----------------------------- private function declarations ----------------------------------------
static void app_iface_on_part_detected(partition_t *partition);
static void app_iface_on_part_unsupported(partition_t *partition);
static void app_iface_on_part_mounting(partition_t *partition);
static void app_iface_on_part_mounted(partition_t *partition);
static void app_iface_on_part_mount_err(partition_t *partition);
static void app_iface_on_part_remounting(partition_t *partition);
static void app_iface_on_part_unmounting(partition_t *partition);
static void app_iface_on_part_unmounted(partition_t *partition);
static void app_iface_on_part_invalid(partition_t *partition);

static void app_iface_on_dev_detected(device_t *device);
static void app_iface_on_dev_nomedia(device_t *device);
static void app_iface_on_dev_automounting(device_t *device);
static void app_iface_on_dev_automounted(device_t *device);
static void app_iface_on_dev_unmounting(device_t *device);
static void app_iface_on_dev_unmounted(device_t *device);
static void app_iface_on_dev_invalid(device_t *device);

static void app_iface_send_part_event_msg(partition_t *partition, info_event_t type);
static void app_iface_send_dev_event_msg(device_t *device, info_event_t type);
//----------------------------------------------------------------------------------------------------

//----------------------------- callback registration ------------------------------------------------
static const app_iface_event_callbacks_t apiface_implv2_callbacks = {
        .on_partition_detected=app_iface_on_part_detected,
        .on_partition_unsupported=app_iface_on_part_unsupported,
        .on_partition_mounting=app_iface_on_part_mounting,
        .on_partition_mounted=app_iface_on_part_mounted,
        .on_partition_mount_err=app_iface_on_part_mount_err,
        .on_partition_remounting=app_iface_on_part_remounting,
        .on_partition_unmounting=app_iface_on_part_unmounting,
        .on_partition_unmounted=app_iface_on_part_unmounted,
        .on_partition_invalid=app_iface_on_part_invalid,
        .on_device_detected=app_iface_on_dev_detected,
        .on_device_nomedia=app_iface_on_dev_nomedia,
        .on_device_automounting=app_iface_on_dev_automounting,
        .on_device_automounted=app_iface_on_dev_automounted,
        .on_device_unmounting=app_iface_on_dev_unmounting,
        .on_device_unmounted=app_iface_on_dev_unmounted,
        .on_device_invalid=app_iface_on_dev_invalid,
};
//----------------------------------------------------------------------------------------------------

//----------------------------- implementation of abstract members of app_iface ----------------------
error_code_t app_iface_init_impl(void)
{
    app_iface_register_event_callbacks(&apiface_implv2_callbacks);
    return filedb_writer_init();
}

void app_iface_deinit_impl(void)
{
	filedb_writer_deinit();
}

void app_iface_handle_message(connection_ctx_t *connection)
{
	message_dispatcher_on_new_message(connection);
}
//----------------------------------------------------------------------------------------------------

//----------------------------- private function definitions -----------------------------------------
static void app_iface_on_part_detected(partition_t *partition)
{
	app_iface_send_part_event_msg(partition, PARTITION_DETECTED_EVENT);
}

static void app_iface_on_part_unsupported(partition_t *partition)
{
	filedb_writer_signal_partition_handled(partition);
	app_iface_send_part_event_msg(partition, PARTITION_UNSUPPORTED_EVENT);
}

static void app_iface_on_part_mounting(partition_t *partition)
{
	app_iface_send_part_event_msg(partition, PARTITION_MOUNTING_EVENT);
}

static void app_iface_on_part_mounted(partition_t *partition)
{
	filedb_writer_signal_partition_handled(partition);
	app_iface_send_part_event_msg(partition, PARTITION_MOUNTED_EVENT);
}

static void app_iface_on_part_mount_err(partition_t *partition)
{
	filedb_writer_signal_partition_handled(partition);
	app_iface_send_part_event_msg(partition, PARTITION_MOUNT_ERR_EVENT);
}

static void app_iface_on_part_remounting(partition_t *partition)
{
	app_iface_send_part_event_msg(partition, PARTITION_REMOUNTING_EVENT);
}

static void app_iface_on_part_unmounting(partition_t *partition)
{
	app_iface_send_part_event_msg(partition, PARTITION_UNMOUNTING_EVENT);
}

static void app_iface_on_part_unmounted(partition_t *partition)
{
	filedb_writer_signal_partition_handled(partition);
	app_iface_send_part_event_msg(partition, PARTITION_UNMOUNTED_EVENT);
}

static void app_iface_on_part_invalid(partition_t *partition)
{
	filedb_writer_signal_partition_invalid(partition);
	app_iface_send_part_event_msg(partition, PARTITION_INVALID_EVENT);
}

static void app_iface_on_dev_detected(device_t *device)
{
	filedb_writer_signal_device_detected(device);
	app_iface_send_dev_event_msg(device, DEVICE_DETECTED_EVENT);
}

static void app_iface_on_dev_nomedia(device_t *device)
{
	filedb_writer_signal_device_handled(device);
	app_iface_send_dev_event_msg(device, DEVICE_NOMEDIA_EVENT);
}

static void app_iface_on_dev_automounting(device_t *device)
{
	app_iface_send_dev_event_msg(device, DEVICE_AUTOMOUNTING_EVENT);
}

static void app_iface_on_dev_automounted(device_t *device)
{
	filedb_writer_signal_device_handled(device);
	app_iface_send_dev_event_msg(device, DEVICE_AUTOMOUNTED_EVENT);
}

static void app_iface_on_dev_unmounting(device_t *device)
{
	app_iface_send_dev_event_msg(device, DEVICE_UNMOUNTING_EVENT);
}

static void app_iface_on_dev_unmounted(device_t *device)
{
	filedb_writer_signal_device_handled(device);
	app_iface_send_dev_event_msg(device, DEVICE_UNMOUNTED_EVENT);
}

static void app_iface_on_dev_invalid(device_t *device)
{
	filedb_writer_signal_device_invalid(device);
	app_iface_send_dev_event_msg(device, DEVICE_INVALID_EVENT);
}

static void app_iface_send_part_event_msg(partition_t *partition, info_event_t type)
{
	partition_info_t part_info;
	device_info_t dev_info;
	app_iface_extract_part_info(&part_info,partition);
	app_iface_extract_dev_info(&dev_info,partition_get_device(partition));
	message_sender_send_partition_event_msg(type,&dev_info,&part_info);
}

static void app_iface_send_dev_event_msg(device_t *device, info_event_t type)
{
	device_info_t dev_info;
	app_iface_extract_dev_info(&dev_info,device);
	message_sender_send_device_event_msg(type,&dev_info);
}
//----------------------------------------------------------------------------------------------------
